<?php
header('Content-Type: application/json');
$response = ['success' => false, 'users' => [], 'pagination' => null, 'message' => 'Gagal memuat data pengguna.'];

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../includes/session.php';
require_once __DIR__ . '/../../lib/Auth.php';
require_once __DIR__ . '/../../includes/functions.php';

if (!Auth::isLoggedIn() || $_SESSION['user_role'] !== 'admin') {
    http_response_code(403);
    $response['message'] = 'Akses ditolak.';
    echo json_encode($response);
    exit;
}

if (!isset($pdo) || !$pdo instanceof PDO) {
    http_response_code(500);
    error_log("PDO object not available in get_admin_users.php");
    $response['message'] = 'Koneksi database gagal.';
    echo json_encode($response);
    exit;
}

try {
    $limit = filter_input(INPUT_GET, 'limit', FILTER_VALIDATE_INT, ['options' => ['default' => 15, 'min_range' => 5, 'max_range' => 100]]);
    $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
    $status = $_GET['status'] ?? 'all';
    $role = $_GET['role'] ?? 'all';
    $sortBy = $_GET['sort_by'] ?? 'u.created_at';
    $sortType = $_GET['sort_type'] ?? 'DESC';
    $searchColumn = $_GET['search_column'] ?? 'u.id';
    $searchKeyword = filter_input(INPUT_GET, 'search_keyword', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';

    $offset = ($page - 1) * $limit;

    $allowedSortColumns = ['u.created_at', 'u.id', 'u.username', 'u.email', 'u.full_name', 'u.balance', 'u.role', 'u.status'];
    $allowedSortTypes = ['ASC', 'DESC'];
    if (!in_array($sortBy, $allowedSortColumns)) { $sortBy = 'u.created_at'; }
    if (!in_array(strtoupper($sortType), $allowedSortTypes)) { $sortType = 'DESC'; }

    $allowedSearchColumns = ['u.id', 'u.username', 'u.email', 'u.full_name', 'u.phone_number'];
    if (!in_array($searchColumn, $allowedSearchColumns)) { $searchColumn = 'u.id'; }

    $sqlSelect = "SELECT u.id, u.username, u.email, u.full_name, u.balance, u.role, u.status, u.created_at, u.phone_number ";
    $sqlFrom = "FROM users u ";
    $sqlWhereBase = "WHERE 1=1 ";
    $params = [];
    $whereClauses = [];

    if ($status !== 'all') {
        $validStatuses = ['active', 'inactive', 'banned'];
        if (in_array($status, $validStatuses)) {
            $whereClauses[] = "u.status = :status";
            $params[':status'] = $status;
        }
    }
     if ($role !== 'all') {
        $validRoles = ['member', 'admin'];
        if (in_array($role, $validRoles)) {
            $whereClauses[] = "u.role = :role";
            $params[':role'] = $role;
        }
    }
    if (!empty($searchKeyword)) {
        if ($searchColumn === 'u.id') {
            if (is_numeric($searchKeyword)) {
                $whereClauses[] = "u.id = :search_keyword_id";
                $params[':search_keyword_id'] = (int)$searchKeyword;
            } else {
                $whereClauses[] = "1 = 0";
            }
        } else {
            $whereClauses[] = $searchColumn . " LIKE :search_keyword";
            $params[':search_keyword'] = '%' . $searchKeyword . '%';
        }
    }

    $sqlWhere = $sqlWhereBase . (!empty($whereClauses) ? " AND " . implode(" AND ", $whereClauses) : "");

    $totalUsers = 0;
    $totalPages = 1;
    $sqlCount = "SELECT COUNT(u.id) " . $sqlFrom . $sqlWhere;
    try {
        $stmtCount = $pdo->prepare($sqlCount);
        $stmtCount->execute($params);
        $totalUsers = (int)$stmtCount->fetchColumn();
        $totalPages = $totalUsers > 0 ? ceil($totalUsers / $limit) : 1;
        if ($page > $totalPages) {
            $page = $totalPages;
            $offset = ($page - 1) * $limit;
        } elseif ($page < 1) {
            $page = 1;
            $offset = 0;
        }
    } catch (\PDOException $e) {
        error_log("Error counting admin users: " . $e->getMessage());
        throw $e;
    }

    $users = [];
    $sqlData = $sqlSelect . $sqlFrom . $sqlWhere . " ORDER BY {$sortBy} {$sortType} LIMIT :limit OFFSET :offset";
    try {
        $stmtData = $pdo->prepare($sqlData);
        foreach ($params as $key => &$value) {
             if ($key === ':search_keyword_id') {
                 $stmtData->bindValue($key, $value, PDO::PARAM_INT);
             } else {
                 $stmtData->bindValue($key, $value);
             }
        }
        unset($value);

        $stmtData->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmtData->bindParam(':offset', $offset, PDO::PARAM_INT);

        $stmtData->execute();
        $users = $stmtData->fetchAll(PDO::FETCH_ASSOC);

        $response['success'] = true;
        $response['users'] = $users;
        $response['pagination'] = [
            'totalItems' => $totalUsers,
            'totalPages' => $totalPages,
            'currentPage' => $page,
            'limit' => $limit,
            'offset' => $offset
        ];
        unset($response['message']);

    } catch (\PDOException $e) {
        error_log("Error fetching admin users data: " . $e->getMessage());
        $response['message'] = "Gagal mengambil data pengguna.";
        http_response_code(500);
    }

} catch (\Throwable $e) {
    error_log("AJAX Get Admin Users General Error: " . $e->getMessage());
    $response['message'] = "Terjadi kesalahan internal server.";
    if (defined('DEBUG_MODE') && DEBUG_MODE === true) { $response['debug_php'] = $e->getMessage(); }
    http_response_code(500);
}

echo json_encode($response);
exit;
?>
