<?php
header('Content-Type: application/json');
$response = ['success' => false, 'orders' => [], 'pagination' => null, 'message' => 'Gagal memuat data pesanan.'];

require_once __DIR__ . '/../../../config/config.php';
require_once __DIR__ . '/../../../includes/session.php';
require_once __DIR__ . '/../../../lib/Auth.php';
require_once __DIR__ . '/../../../includes/functions.php';

if (!Auth::isLoggedIn() || $_SESSION['user_role'] !== 'admin') {
    http_response_code(403);
    $response['message'] = 'Akses ditolak.';
    echo json_encode($response);
    exit;
}

if (!isset($pdo) || !$pdo instanceof PDO) {
    http_response_code(500);
    error_log("PDO object not available in get_admin_orders.php");
    $response['message'] = 'Koneksi database gagal.';
    echo json_encode($response);
    exit;
}

try {
    $limit = filter_input(INPUT_GET, 'limit', FILTER_VALIDATE_INT, ['options' => ['default' => 15, 'min_range' => 5, 'max_range' => 100]]);
    $page = filter_input(INPUT_GET, 'page', FILTER_VALIDATE_INT, ['options' => ['default' => 1, 'min_range' => 1]]);
    $status = $_GET['status'] ?? 'all';
    $sortBy = $_GET['sort_by'] ?? 'o.created_at';
    $sortType = $_GET['sort_type'] ?? 'DESC';
    $searchColumn = $_GET['search_column'] ?? 'o.id';
    $searchKeyword = filter_input(INPUT_GET, 'search_keyword', FILTER_SANITIZE_SPECIAL_CHARS) ?? '';
    $dateStart = $_GET['date_start'] ?? '';
    $dateEnd = $_GET['date_end'] ?? '';
    $userId = filter_input(INPUT_GET, 'user_id', FILTER_VALIDATE_INT);
    $serviceId = filter_input(INPUT_GET, 'service_id', FILTER_VALIDATE_INT);

    $offset = ($page - 1) * $limit;

    $allowedSortColumns = ['o.created_at', 'o.id', 's.name', 'o.price', 'o.quantity', 'o.status', 'u.username'];
    $allowedSortTypes = ['ASC', 'DESC'];
    if (!in_array($sortBy, $allowedSortColumns)) { $sortBy = 'o.created_at'; }
    if (!in_array(strtoupper($sortType), $allowedSortTypes)) { $sortType = 'DESC'; }

    $allowedSearchColumns = ['o.id', 'o.target', 's.name', 'u.username'];
    if (!in_array($searchColumn, $allowedSearchColumns)) { $searchColumn = 'o.id'; }

    $sqlSelect = "SELECT o.id, o.user_id, o.service_id, o.target, o.quantity, o.price, o.status, o.start_count, o.remains, o.order_source, o.api_order_id, o.created_at, o.updated_at, u.username, s.name as service_name ";
    $sqlFrom = "FROM orders o LEFT JOIN users u ON o.user_id = u.id LEFT JOIN services s ON o.service_id = s.id ";
    $sqlWhereBase = "WHERE 1=1 ";
    $params = [];
    $whereClauses = [];

    if ($status !== 'all') {
        $validStatuses = ['pending', 'processing', 'in progress', 'success', 'error', 'partial', 'canceled'];
        if (in_array($status, $validStatuses)) {
            $whereClauses[] = "o.status = :status";
            $params[':status'] = $status;
        }
    }
    if (!empty($searchKeyword)) {
        if ($searchColumn === 'o.id') {
            if (is_numeric($searchKeyword)) {
                $whereClauses[] = "o.id = :search_keyword_id";
                $params[':search_keyword_id'] = (int)$searchKeyword;
            } else {
                $whereClauses[] = "1 = 0";
            }
        } else {
            $whereClauses[] = $searchColumn . " LIKE :search_keyword";
            $params[':search_keyword'] = '%' . $searchKeyword . '%';
        }
    }
     if (!empty($dateStart)) {
        $whereClauses[] = "DATE(o.created_at) >= :date_start";
        $params[':date_start'] = $dateStart;
    }
    if (!empty($dateEnd)) {
        $whereClauses[] = "DATE(o.created_at) <= :date_end";
        $params[':date_end'] = $dateEnd;
    }
    if ($userId) {
        $whereClauses[] = "o.user_id = :user_id";
        $params[':user_id'] = $userId;
    }
    if ($serviceId) {
        $whereClauses[] = "o.service_id = :service_id";
        $params[':service_id'] = $serviceId;
    }

    $sqlWhere = $sqlWhereBase . (!empty($whereClauses) ? " AND " . implode(" AND ", $whereClauses) : "");

    $totalOrders = 0;
    $totalPages = 1;
    $sqlCount = "SELECT COUNT(o.id) " . $sqlFrom . $sqlWhere;
    try {
        $stmtCount = $pdo->prepare($sqlCount);
        $stmtCount->execute($params);
        $totalOrders = (int)$stmtCount->fetchColumn();
        $totalPages = $totalOrders > 0 ? ceil($totalOrders / $limit) : 1;
        if ($page > $totalPages) {
            $page = $totalPages;
            $offset = ($page - 1) * $limit;
        } elseif ($page < 1) {
            $page = 1;
            $offset = 0;
        }
    } catch (\PDOException $e) {
        error_log("Error counting admin orders: " . $e->getMessage());
        throw $e;
    }

    $orders = [];
    $sqlData = $sqlSelect . $sqlFrom . $sqlWhere . " ORDER BY {$sortBy} {$sortType} LIMIT :limit OFFSET :offset";
    try {
        $stmtData = $pdo->prepare($sqlData);
        foreach ($params as $key => &$value) {
             if ($key === ':search_keyword_id' || $key === ':user_id' || $key === ':service_id') {
                 $stmtData->bindValue($key, $value, PDO::PARAM_INT);
             } else {
                 $stmtData->bindValue($key, $value);
             }
        }
        unset($value);

        $stmtData->bindParam(':limit', $limit, PDO::PARAM_INT);
        $stmtData->bindParam(':offset', $offset, PDO::PARAM_INT);

        $stmtData->execute();
        $orders = $stmtData->fetchAll(PDO::FETCH_ASSOC);

        $response['success'] = true;
        $response['orders'] = $orders;
        $response['pagination'] = [
            'totalItems' => $totalOrders,
            'totalPages' => $totalPages,
            'currentPage' => $page,
            'limit' => $limit,
            'offset' => $offset
        ];
        unset($response['message']);

    } catch (\PDOException $e) {
        error_log("Error fetching admin orders data: " . $e->getMessage());
        $response['message'] = "Gagal mengambil data pesanan.";
        http_response_code(500);
    }

} catch (\Throwable $e) {
    error_log("AJAX Get Admin Orders General Error: " . $e->getMessage());
    $response['message'] = "Terjadi kesalahan internal server.";
    if (defined('DEBUG_MODE') && DEBUG_MODE === true) { $response['debug_php'] = $e->getMessage(); }
    http_response_code(500);
}

echo json_encode($response);
exit;
?>
